
import os
import tkinter as tk
from tkinter import Canvas, Label
import requests
from PIL import Image, ImageTk
import io
import cv2
import threading
import time
import logging

# Setup logging
LOG_FILE = "raspberry_slider_app.log"
logging.basicConfig(filename=LOG_FILE, level=logging.INFO, format="%(asctime)s - %(levelname)s - %(message)s")

brachid = "<?php
print($_GET['id'])
?>"
SERVICE_URL = "https://manafez.gt4it.com/api/v1/place/<?php
print($_GET['id'])
?>"
UPLOAD_URL = "https://manafez.gt4it.com/api/v1/recordings"

SLIDE_INTERVAL = 5  # Seconds per slide
RELOAD_INTERVAL = 3600  # Fetch new images every 10 minutes
CAPTURE_INTERVAL = 3600000  # Capture every 60 seconds
IMAGE_PATH = "capture.jpg"

class RaspberrySliderApp:
    def __init__(self, root):
        self.root = root
        self.root.configure(bg='black')
        self.root.attributes('-fullscreen', True)

        # Get screen size
        self.screen_width = self.root.winfo_screenwidth()
        self.screen_height = self.root.winfo_screenheight()
        self.root.geometry(f"{self.screen_width}x{self.screen_height}+0+0")

        # Canvas for full-screen images
        self.canvas = Canvas(self.root, bg='black', highlightthickness=0)
        self.canvas.pack(fill=tk.BOTH, expand=True)

        # Error message label
        self.error_label = Label(self.root, text="", fg="red", bg="black", font=("Arial", 20))
        self.error_label.place(relx=0.5, rely=0.5, anchor="center")

        self.images = []
        self.current_index = 0
        self.internet_available = self.is_internet_available()

        # Start processes
        self.check_internet_and_start()

        self.root.bind('<Escape>', lambda e: self.exit_fullscreen())

    def is_internet_available(self):
        """Check for internet connectivity."""
        try:
            requests.get("https://www.google.com", timeout=5)
            return True
        except requests.RequestException:
            return False

    def check_internet_and_start(self):
        """Check for internet and start processes accordingly."""
        if self.is_internet_available():
            self.clear_error()
            self.reload_images()
            self.start_slideshow()
            self.start_camera_capture()
            self.schedule_camera_capture()
        else:
            self.show_error("No Internet. Retrying in 10 seconds...")
            logging.warning("No Internet connection. Retrying in 10 seconds...")
            self.root.after(10000, self.check_internet_and_start)

    def show_error(self, message):
        """Display error messages."""
        self.error_label.config(text=message)
        logging.error(message)
        self.error_label.after(10000, self.clear_error)

    def clear_error(self):
        """Clear error messages."""
        self.error_label.config(text="")

    def reload_images(self):
        """Fetch images from the API."""
        if not self.is_internet_available():
            self.show_error("No Internet. Retrying...")
            logging.warning("No Internet. Retrying to fetch images...")
            self.root.after(RELOAD_INTERVAL * 1000, self.reload_images)
            return

        try:
            response = requests.get(SERVICE_URL, timeout=10)
            response.raise_for_status()
            data = response.json()
            new_images = data if isinstance(data, list) else data.get("images", [])

            if not new_images:
                self.show_error("No images available. Retrying...")
                logging.warning("No images available from the API.")
            else:
                self.clear_error()
                self.images = new_images
                self.current_index = 0
                logging.info(f"Successfully fetched {len(new_images)} images.")

        except Exception as e:
            self.show_error("Error fetching images.")
            logging.error(f"Error fetching images: {e}")

        self.root.after(RELOAD_INTERVAL * 1000, self.reload_images)

    def start_slideshow(self):
        """Display images in a loop."""
        if not self.images:
            self.root.after(SLIDE_INTERVAL * 1000, self.start_slideshow)
            return

        self.show_image(self.images[self.current_index])
        self.current_index = (self.current_index + 1) % len(self.images)
        self.root.after(SLIDE_INTERVAL * 1000, self.start_slideshow)

    def show_image(self, image_url):
        """Load and display an image in fullscreen mode."""
        try:
            pil_image = self.load_image(image_url)
            tk_image = ImageTk.PhotoImage(pil_image)

            # Clear previous images
            self.canvas.delete("all")

            # Display centered full-screen image
            self.canvas.create_image(self.screen_width // 2, self.screen_height // 2, anchor=tk.CENTER, image=tk_image)
            self.canvas.image = tk_image  # Keep a reference to prevent garbage collection

        except Exception as e:
            self.show_error(f"Error loading image: {e}")
            logging.error(f"Error loading image: {e}")

    def load_image(self, url):
        """Fetch and resize an image to fullscreen."""
        response = requests.get(url, timeout=10)
        response.raise_for_status()
        img_data = io.BytesIO(response.content)
        pil_image = Image.open(img_data)
        pil_image = pil_image.resize((self.screen_width, self.screen_height), Image.LANCZOS)
        return pil_image

    def schedule_camera_capture(self):
        """Schedule camera capture if internet is available."""
        if self.is_internet_available():
            self.root.after(CAPTURE_INTERVAL, self.start_camera_capture)
        else:
            self.show_error("No Internet. Camera capture paused.")
            logging.warning("No Internet. Camera capture paused.")
            self.root.after(10000, self.schedule_camera_capture)  # Retry in 10 seconds

    def start_camera_capture(self):
        """Capture an image and upload it in a separate thread."""
        threading.Thread(target=self.capture_and_upload, daemon=True).start()
        self.schedule_camera_capture()

    def capture_and_upload(self):
        """Capture and upload an image if internet is available."""
        if not self.is_internet_available():
            logging.warning("No Internet. Skipping capture.")
            return

        cap = cv2.VideoCapture(0)
        if not cap.isOpened():
            logging.error("Error: Could not open camera.")
            return

        ret, frame = cap.read()
        cap.release()

        if ret:
            cv2.imwrite(IMAGE_PATH, frame)
            time.sleep(1)  # Ensure file is saved before upload
            self.upload_image(IMAGE_PATH)
        else:
            logging.error("Error: Failed to capture image.")

    def upload_image(self, file_path):
        """Upload an image if internet is available."""
        if not self.is_internet_available():
            self.show_error("No Internet. Image upload paused.")
            logging.warning("No Internet. Image upload paused.")
            return
        
        if not os.path.exists(file_path):
            logging.error(f"Error: File {file_path} does not exist.")
            return
        
        logging.info(f"Uploading image: {file_path}...")
        with open(file_path, 'rb') as image_file:
            files = {'image': image_file}
            data = {'type': 1, 'place_id': brachid}
            try:
                response = requests.post(UPLOAD_URL, files=files, data=data)
                logging.info(f"Upload Response: {response.status_code} - {response.text}")
                if response.status_code == 200:
                    os.remove(file_path)
                    logging.info(f"Deleted uploaded image: {file_path}")
            except Exception as e:
                logging.error(f"Failed to upload image: {e}")

    def exit_fullscreen(self):
        """Exit fullscreen and close the application."""
        self.root.attributes('-fullscreen', False)
        self.root.destroy()

if __name__ == "__main__":
    root = tk.Tk()
    app = RaspberrySliderApp(root)
    root.mainloop()
