import sys
import requests
from PIL import Image
from io import BytesIO

def is_tv_on(image_url, tv_area=None, brightness_threshold=50):
    """
    Determines if the TV is on or off by analyzing the brightness of the specified TV screen area.

    Args:
        image_url (str): URL of the image.
        tv_area (tuple): Coordinates of the TV screen (x, y, width, height). Default is the whole image.
        brightness_threshold (int): Brightness threshold to classify TV as "on".

    Returns:
        bool: True if the TV is on, False if it's off.
    """
    try:
        # Download the image
        response = requests.get(image_url)
        response.raise_for_status()  # Ensure the request was successful

        # Open the image
        img = Image.open(BytesIO(response.content))
        img = img.convert("RGB")  # Ensure image is in RGB mode
        width, height = img.size

        # Default to the entire image if no TV area is specified
        if not tv_area:
            tv_area = (0, 0, width, height)

        x, y, tv_width, tv_height = tv_area

        # Crop the TV area
        cropped_img = img.crop((x, y, x + tv_width, y + tv_height))

        # Get pixel data
        pixels = list(cropped_img.getdata())

        # Calculate average brightness
        total_brightness = 0
        for r, g, b in pixels:
            brightness = (r + g + b) / 3  # Brightness is the average of R, G, B
            total_brightness += brightness

        average_brightness = total_brightness / len(pixels)

        # Determine if the TV is on
        return average_brightness > brightness_threshold

    except Exception as e:
        print(f"Error processing the image: {e}")
        return False

# Main function to handle command-line input
if __name__ == "__main__":
    if len(sys.argv) < 2:
        print("Usage: python detect_tv.py <image_url>")
        sys.exit(1)

    image_url = sys.argv[1]  # Get image URL from command-line argument
    tv_area = (50, 50, 600, 300)  # Adjust to the TV screen coordinates (x, y, width, height)
    brightness_threshold = 50  # Define a suitable brightness threshold

    if is_tv_on(image_url, tv_area, brightness_threshold):
        print("The TV is ON.")
    else:
        print("The TV is OFF.")
